'use client';

import { useState, useEffect } from 'react';

export default function Home() {
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [formData, setFormData] = useState({
    name: '',
    phone: '',
    email: '',
    message: ''
  });
  const [isFormSubmitted, setIsFormSubmitted] = useState(false);

  // Scroll animasyon efekti
  useEffect(() => {
    const handleScroll = () => {
      const elements = document.querySelectorAll('.scroll-animate');
      elements.forEach((element) => {
        const elementTop = element.getBoundingClientRect().top;
        const elementVisible = 150;
        
        if (elementTop < window.innerHeight - elementVisible) {
          element.classList.add('animate-fade-in');
        }
      });
    };

    window.addEventListener('scroll', handleScroll);
    return () => window.removeEventListener('scroll', handleScroll);
  }, []);

  // Form validation ve submit
  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!formData.name || !formData.phone || !formData.message) {
      alert('Lütfen tüm alanları doldurunuz.');
      return;
    }

    if (formData.phone.length < 10) {
      alert('Lütfen geçerli bir telefon numarası giriniz.');
      return;
    }

    // Form gönderim simülasyonu
    setIsFormSubmitted(true);
    setFormData({ name: '', phone: '', email: '', message: '' });
    
    setTimeout(() => {
      setIsFormSubmitted(false);
    }, 3000);
  };

  const handleInputChange = (e: React.ChangeEvent<HTMLInputElement | HTMLTextAreaElement>) => {
    setFormData({
      ...formData,
      [e.target.name]: e.target.value
    });
  };

  return (
    <div className="min-h-screen bg-white">
      {/* Header */}
      <header className="bg-white shadow-md fixed w-full top-0 z-50">
        <div className="container mx-auto px-4 py-3">
          <div className="flex justify-between items-center">
            <div className="text-2xl font-bold text-blue-600">
              Çartes Tesisat
            </div>
            
            {/* Desktop Menu */}
            <nav className="hidden md:flex space-x-8">
              <a href="#anasayfa" className="text-gray-700 hover:text-blue-600 transition-colors cursor-pointer">Ana Sayfa</a>
              <a href="#hizmetler" className="text-gray-700 hover:text-blue-600 transition-colors cursor-pointer">Hizmetler</a>
              <a href="#hakkimizda" className="text-gray-700 hover:text-blue-600 transition-colors cursor-pointer">Hakkımızda</a>
              <a href="#iletisim" className="text-gray-700 hover:text-blue-600 transition-colors cursor-pointer">İletişim</a>
            </nav>

            {/* Mobile Menu Button */}
            <button 
              className="md:hidden"
              onClick={() => setIsMenuOpen(!isMenuOpen)}
            >
              <div className="w-6 h-0.5 bg-gray-700 mb-1"></div>
              <div className="w-6 h-0.5 bg-gray-700 mb-1"></div>
              <div className="w-6 h-0.5 bg-gray-700"></div>
            </button>
          </div>

          {/* Mobile Menu */}
          {isMenuOpen && (
            <nav className="md:hidden mt-4 pb-4">
              <a href="#anasayfa" className="block py-2 text-gray-700 hover:text-blue-600 cursor-pointer" onClick={() => setIsMenuOpen(false)}>Ana Sayfa</a>
              <a href="#hizmetler" className="block py-2 text-gray-700 hover:text-blue-600 cursor-pointer" onClick={() => setIsMenuOpen(false)}>Hizmetler</a>
              <a href="#hakkimizda" className="block py-2 text-gray-700 hover:text-blue-600 cursor-pointer" onClick={() => setIsMenuOpen(false)}>Hakkımızda</a>
              <a href="#iletisim" className="block py-2 text-gray-700 hover:text-blue-600 cursor-pointer" onClick={() => setIsMenuOpen(false)}>İletişim</a>
            </nav>
          )}
        </div>
      </header>

      {/* Hero Section */}
      <section id="anasayfa" className="pt-20 bg-gradient-to-br from-blue-50 to-blue-100 min-h-screen flex items-center">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div className="text-center md:text-left">
              <h1 className="text-4xl md:text-6xl font-bold text-gray-800 mb-6">
                Niğde'nin En <span className="text-blue-600">Güvenilir</span> Tesisatçısı
              </h1>
              <p className="text-xl text-gray-600 mb-8">
                Niğde'de sıhhi tesisat, su kaçak tespiti, petek temizliği ve acil tesisatçı hizmetleri. 
                7/24 acil müdahale ile evinizin tesisat sorunlarını çözüyoruz.
              </p>
              <div className="flex flex-col sm:flex-row gap-4 justify-center md:justify-start">
                <a 
                  href="tel:+905551234567" 
                  className="bg-blue-600 text-white px-8 py-4 rounded-lg text-lg font-semibold hover:bg-blue-700 transition-colors text-center cursor-pointer whitespace-nowrap"
                >
                  📞 Hemen Ara
                </a>
                <a 
                  href="#iletisim" 
                  className="border-2 border-blue-600 text-blue-600 px-8 py-4 rounded-lg text-lg font-semibold hover:bg-blue-600 hover:text-white transition-colors text-center cursor-pointer whitespace-nowrap"
                >
                  İletişime Geç
                </a>
              </div>
            </div>
            <div className="text-center">
              <img 
                src="https://readdy.ai/api/search-image?query=Professional%20plumber%20working%20with%20pipes%20and%20tools%20in%20modern%20bathroom%20setting%2C%20clean%20background%20with%20plumbing%20equipment%2C%20realistic%20professional%20photography%20style%2C%20bright%20lighting%2C%20turkish%20plumber%20in%20work%20uniform%20fixing%20pipes&width=600&height=400&seq=hero1&orientation=landscape" 
                alt="Niğde Çartes Tesisat - Profesyonel Su Tesisatçısı"
                className="rounded-lg shadow-xl w-full object-cover object-top"
              />
            </div>
          </div>
        </div>
      </section>

      {/* Hizmetler Section */}
      <section id="hizmetler" className="py-20 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16 scroll-animate">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-800 mb-4">
              Niğde'de Tesisat Hizmetlerimiz
            </h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Niğde'de tesisatçı mı arıyorsunuz? Çartes Tesisat olarak her türlü sıhhi tesisat sorununuza profesyonel çözüm sunuyoruz.
            </p>
          </div>

          <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-8">
            {[
              {
                title: "Su Kaçak Tespiti",
                description: "Niğde'de su kaçağı tespiti ve onarımı. Modern cihazlarla hızlı tespit, profesyonel onarım.",
                icon: "💧",
                image: "https://readdy.ai/api/search-image?query=Water%20leak%20detection%20equipment%20and%20tools%2C%20professional%20plumber%20using%20detection%20device%2C%20modern%20bathroom%20setting%2C%20realistic%20photography%2C%20clean%20background&width=400&height=300&seq=service1&orientation=landscape"
              },
              {
                title: "Petek Temizliği",
                description: "Kalorifer tesisatı ve petek temizlik hizmetleri. Niğde'de kombi bağlantısı ve kalorifer hattı döşeme.",
                icon: "🔧",
                image: "https://readdy.ai/api/search-image?query=Radiator%20cleaning%20service%2C%20heating%20system%20maintenance%2C%20professional%20plumber%20cleaning%20radiator%2C%20modern%20interior%2C%20clean%20background&width=400&height=300&seq=service2&orientation=landscape"
              },
              {
                title: "Acil Tesisat",
                description: "7/24 acil tesisatçı Niğde. Musluk tamiri, lavabo tıkanıklığı, su tesisatı acil müdahale.",
                icon: "🚨",
                image: "https://readdy.ai/api/search-image?query=Emergency%20plumbing%20service%2C%20plumber%20fixing%20urgent%20pipe%20problem%2C%20professional%20emergency%20repair%2C%20modern%20tools%20and%20equipment&width=400&height=300&seq=service3&orientation=landscape"
              },
              {
                title: "Banyo Tadilatı",
                description: "Mutfak ve banyo tadilatı hizmetleri. Niğde'de sıhhi tesisat yenileme ve tadilat işleri.",
                icon: "🛁",
                image: "https://readdy.ai/api/search-image?query=Bathroom%20renovation%20project%2C%20modern%20bathroom%20installation%2C%20professional%20plumber%20installing%20fixtures%2C%20clean%20modern%20design&width=400&height=300&seq=service4&orientation=landscape"
              },
              {
                title: "Kombi Servisi",
                description: "Kombi bağlantısı, bakımı ve onarımı. Niğde'de kombi tesisatı uzmanı hizmetleri.",
                icon: "🔥",
                image: "https://readdy.ai/api/search-image?query=Boiler%20installation%20and%20maintenance%2C%20heating%20system%20repair%2C%20professional%20technician%20working%20on%20boiler%2C%20modern%20equipment&width=400&height=300&seq=service5&orientation=landscape"
              },
              {
                title: "Tıkanıklık Açma",
                description: "Lavabo, klozet, gider tıkanıklığı açma. Niğde'de tesisat tıkanıklık açma uzmanı.",
                icon: "🌊",
                image: "https://readdy.ai/api/search-image?query=Drain%20cleaning%20service%2C%20professional%20plumber%20clearing%20blocked%20drain%2C%20modern%20plumbing%20tools%2C%20clean%20bathroom%20setting&width=400&height=300&seq=service6&orientation=landscape"
              }
            ].map((service, index) => (
              <div key={index} className="bg-white rounded-lg shadow-lg overflow-hidden hover:shadow-xl transition-shadow scroll-animate">
                <img 
                  src={service.image} 
                  alt={service.title}
                  className="w-full h-48 object-cover object-top"
                />
                <div className="p-6">
                  <div className="text-4xl mb-4">{service.icon}</div>
                  <h3 className="text-xl font-semibold text-gray-800 mb-3">{service.title}</h3>
                  <p className="text-gray-600">{service.description}</p>
                </div>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Hakkımızda Section */}
      <section id="hakkimizda" className="py-20 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-2 gap-12 items-center">
            <div className="scroll-animate">
              <h2 className="text-3xl md:text-4xl font-bold text-gray-800 mb-6">
                Niğde'de Güvenilir Tesisat Çözümleri
              </h2>
              <p className="text-gray-600 mb-6">
                Çartes Tesisat olarak Niğde Merkez'de yıllardır kaliteli sıhhi tesisat hizmetleri sunuyoruz. 
                Deneyimli ekibimiz ve modern ekipmanlarımızla her türlü tesisat sorununuza hızlı çözüm üretiyoruz.
              </p>
              <div className="space-y-4">
                <div className="flex items-center">
                  <div className="w-2 h-2 bg-blue-600 rounded-full mr-3"></div>
                  <span className="text-gray-700">7/24 Acil Müdahale Hizmeti</span>
                </div>
                <div className="flex items-center">
                  <div className="w-2 h-2 bg-blue-600 rounded-full mr-3"></div>
                  <span className="text-gray-700">Profesyonel ve Deneyimli Ekip</span>
                </div>
                <div className="flex items-center">
                  <div className="w-2 h-2 bg-blue-600 rounded-full mr-3"></div>
                  <span className="text-gray-700">Uygun Fiyat Garantisi</span>
                </div>
                <div className="flex items-center">
                  <div className="w-2 h-2 bg-blue-600 rounded-full mr-3"></div>
                  <span className="text-gray-700">Kaliteli Malzeme Kullanımı</span>
                </div>
              </div>
            </div>
            <div className="text-center scroll-animate">
              <img 
                src="https://readdy.ai/api/search-image?query=Professional%20plumbing%20team%20at%20work%2C%20experienced%20plumbers%20with%20modern%20tools%2C%20working%20in%20clean%20modern%20setting%2C%20teamwork%20and%20professionalism&width=600&height=400&seq=about1&orientation=landscape" 
                alt="Çartes Tesisat Ekibi - Niğde Tesisatçı"
                className="rounded-lg shadow-xl w-full object-cover object-top"
              />
            </div>
          </div>
        </div>
      </section>

      {/* İletişim Section */}
      <section id="iletisim" className="py-20 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-16 scroll-animate">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-800 mb-4">
              İletişime Geçin
            </h2>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Niğde'de acil tesisatçı mı arıyorsunuz? Hemen bize ulaşın, sorununuzu çözelim.
            </p>
          </div>

          <div className="grid md:grid-cols-2 gap-12">
            <div className="scroll-animate">
              <div className="bg-blue-50 rounded-lg p-8">
                <h3 className="text-2xl font-bold text-gray-800 mb-6">İletişim Bilgileri</h3>
                
                <div className="space-y-4 mb-8">
                  <div className="flex items-center">
                    <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center mr-4">
                      <span className="text-white text-xl">📞</span>
                    </div>
                    <div>
                      <p className="font-semibold text-gray-800">Telefon</p>
                      <a href="tel:+905551234567" className="text-blue-600 hover:text-blue-800 cursor-pointer">
                        +90 555 123 45 67
                      </a>
                    </div>
                  </div>
                  
                  <div className="flex items-center">
                    <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center mr-4">
                      <span className="text-white text-xl">📧</span>
                    </div>
                    <div>
                      <p className="font-semibold text-gray-800">E-posta</p>
                      <a href="mailto:info@cartestesisat.com" className="text-blue-600 hover:text-blue-800 cursor-pointer">
                        info@cartestesisat.com
                      </a>
                    </div>
                  </div>
                  
                  <div className="flex items-center">
                    <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center mr-4">
                      <span className="text-white text-xl">📍</span>
                    </div>
                    <div>
                      <p className="font-semibold text-gray-800">Adres</p>
                      <p className="text-gray-600">Niğde Merkez, Niğde</p>
                    </div>
                  </div>
                  
                  <div className="flex items-center">
                    <div className="w-12 h-12 bg-blue-600 rounded-full flex items-center justify-center mr-4">
                      <span className="text-white text-xl">⏰</span>
                    </div>
                    <div>
                      <p className="font-semibold text-gray-800">Çalışma Saatleri</p>
                      <p className="text-gray-600">7/24 Acil Müdahale</p>
                    </div>
                  </div>
                </div>

                <a 
                  href="tel:+905551234567" 
                  className="w-full bg-blue-600 text-white py-4 px-6 rounded-lg text-lg font-semibold hover:bg-blue-700 transition-colors text-center block cursor-pointer whitespace-nowrap"
                >
                  🚨 Tıkla ve Hemen Ara
                </a>
              </div>
            </div>

            <div className="scroll-animate">
              <form onSubmit={handleSubmit} className="bg-gray-50 rounded-lg p-8">
                <h3 className="text-2xl font-bold text-gray-800 mb-6">Mesaj Gönder</h3>
                
                {isFormSubmitted && (
                  <div className="bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4">
                    Mesajınız başarıyla gönderildi! En kısa sürede size dönüş yapacağız.
                  </div>
                )}
                
                <div className="space-y-4">
                  <div>
                    <label className="block text-gray-700 font-medium mb-2">Ad Soyad *</label>
                    <input
                      type="text"
                      name="name"
                      value={formData.name}
                      onChange={handleInputChange}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                      required
                    />
                  </div>
                  
                  <div>
                    <label className="block text-gray-700 font-medium mb-2">Telefon *</label>
                    <input
                      type="tel"
                      name="phone"
                      value={formData.phone}
                      onChange={handleInputChange}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                      required
                    />
                  </div>
                  
                  <div>
                    <label className="block text-gray-700 font-medium mb-2">E-posta</label>
                    <input
                      type="email"
                      name="email"
                      value={formData.email}
                      onChange={handleInputChange}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                    />
                  </div>
                  
                  <div>
                    <label className="block text-gray-700 font-medium mb-2">Mesajınız *</label>
                    <textarea
                      name="message"
                      value={formData.message}
                      onChange={handleInputChange}
                      rows={4}
                      className="w-full px-4 py-3 border border-gray-300 rounded-lg focus:outline-none focus:border-blue-500"
                      required
                    ></textarea>
                  </div>
                  
                  <button
                    type="submit"
                    className="w-full bg-blue-600 text-white py-4 px-6 rounded-lg text-lg font-semibold hover:bg-blue-700 transition-colors cursor-pointer whitespace-nowrap"
                  >
                    Mesaj Gönder
                  </button>
                </div>
              </form>
            </div>
          </div>
        </div>
      </section>

      {/* Footer */}
      <footer className="bg-gray-800 text-white py-12">
        <div className="container mx-auto px-4">
          <div className="grid md:grid-cols-3 gap-8">
            <div>
              <h3 className="text-2xl font-bold mb-4">Çartes Tesisat</h3>
              <p className="text-gray-300 mb-4">
                Niğde'de sıhhi tesisat, su kaçak tespiti, petek temizliği ve acil tesisatçı hizmetleri.
              </p>
              <p className="text-gray-300">
                Niğde'de tesisatçı mı arıyorsunuz? Güvenilir ve profesyonel hizmet için bizi tercih edin.
              </p>
            </div>
            
            <div>
              <h4 className="text-xl font-semibold mb-4">Hizmetlerimiz</h4>
              <ul className="space-y-2 text-gray-300">
                <li>• Su kaçak tespiti Niğde</li>
                <li>• Petek temizliği</li>
                <li>• Acil tesisatçı Niğde</li>
                <li>• Kalorifer tesisatı</li>
                <li>• Musluk tamiri Niğde</li>
                <li>• Banyo tadilatı</li>
              </ul>
            </div>
            
            <div>
              <h4 className="text-xl font-semibold mb-4">İletişim</h4>
              <div className="space-y-2 text-gray-300">
                <p>📞 <a href="tel:+905551234567" className="hover:text-blue-400 cursor-pointer">+90 555 123 45 67</a></p>
                <p>📧 <a href="mailto:info@cartestesisat.com" className="hover:text-blue-400 cursor-pointer">info@cartestesisat.com</a></p>
                <p>📍 Niğde Merkez, Niğde</p>
                <p>⏰ 7/24 Acil Müdahale</p>
              </div>
            </div>
          </div>
          
          <div className="border-t border-gray-700 mt-8 pt-8 text-center text-gray-400">
            <p>&copy; 2024 Çartes Tesisat. Tüm hakları saklıdır. Niğde sıhhi tesisat ve tesisatçı hizmetleri.</p>
          </div>
        </div>
      </footer>

      <style jsx>{`
        .scroll-animate {
          opacity: 0;
          transform: translateY(30px);
          transition: all 0.6s ease;
        }
        
        .animate-fade-in {
          opacity: 1;
          transform: translateY(0);
        }
        
        html {
          scroll-behavior: smooth;
        }
      `}</style>
    </div>
  );
}